<?php
/* --------------------------------------------------------------
   CreateWithdrawalsAction.php 2020-04-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Api\Withdrawal\Actions;

use Gambio\Admin\Withdrawal\Interfaces\WithdrawalId;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalService;
use Gambio\Admin\Withdrawal\Models\Withdrawal;
use Gambio\Admin\Withdrawal\Models\Withdrawals;
use Gambio\Api\Application\Responses\CreateApiMetaDataTrait;
use Gambio\Api\Withdrawal\WithdrawalApiRequestParser;
use Gambio\Api\Withdrawal\WithdrawalApiRequestValidator;
use InvalidArgumentException;
use Slim\Http\Response;
use Slim\Http\ServerRequest as Request;

/**
 * Class CreateWithdrawalsAction
 *
 * @package Gambio\Api\Withdrawal\Actions
 */
class CreateWithdrawalsAction
{
    use CreateApiMetaDataTrait;
    
    /**
     * @var WithdrawalApiRequestParser
     */
    private $requestParser;
    
    /**
     * @var WithdrawalApiRequestValidator
     */
    private $requestValidator;
    
    /**
     * @var WithdrawalService
     */
    private $service;
    
    
    /**
     * CreateWithdrawalsAction constructor.
     *
     * @param WithdrawalApiRequestParser    $requestParser
     * @param WithdrawalApiRequestValidator $requestValidator
     * @param WithdrawalService             $service
     */
    public function __construct(
        WithdrawalApiRequestParser $requestParser,
        WithdrawalApiRequestValidator $requestValidator,
        WithdrawalService $service
    ) {
        $this->requestParser    = $requestParser;
        $this->requestValidator = $requestValidator;
        $this->service          = $service;
    }
    
    
    /**
     * @param Request  $request
     * @param Response $response
     * @param array    $args
     *
     * @return Response
     */
    public function __invoke(Request $request, Response $response, array $args): Response
    {
        $referencedErrors = $this->requestValidator->validatePostRequestBody($request->getParsedBody());
        if (count($referencedErrors) > 0) {
            return $response->withStatus(400)->withJson(['errors' => $referencedErrors]);
        }
        
        $withdrawals = [];
        foreach ($request->getParsedBody() as $reference => $details) {
            try {
                $withdrawals[$reference] = Withdrawal::createFromArray($details);
            } catch (InvalidArgumentException $exception) {
                $referencedErrors[$reference][] = $exception->getMessage();
            }
        }
        if (count($referencedErrors) > 0) {
            return $response->withStatus(422)->withJson(['errors' => $referencedErrors]);
        }
        
        $ids   = $this->service->storeWithdrawals(Withdrawals::createWithReferences($withdrawals));
        $links = [];
        /** @var WithdrawalId $id */
        foreach ($ids as $reference => $id) {
            $links[(string)$reference] = rtrim($this->requestParser->getResourceUrlFromRequest($request), '/')
                                         . '/withdrawals/' . $id->value();
        }
        
        $metaData = $this->createApiMetaData($links);
        
        return $response->withJson([
                                       'data'  => $ids,
                                       '_meta' => $metaData,
                                   ]);
    }
}